<?php
//==================================================================+
// File name   : fpdf.php                                            
// Begin       : 2002-08-03                                          
// Last Update : 2004-06-11                                          
// Author      : Olivier PLATHEY                                     
// Version     : 1.52.1                                              
// License     : Freeware - You may use and modify this              
//               software as you wish                                
//                                                                   
// Description : FPDF is a PHP class which allows to generate        
//               PDF files with straight PHP, that is to say         
//               without using the PDFlib library.                   
//                                                                   
// NOTE:                                                             
// This version contains changes by Nicola Asuni (info@tecnick.com): 
//  - code style and formatting                                      
//  - automatic php documentation in PhpDocumentor Style             
//    (www.phpdoc.org)                                               
//  - all ISO page formats                                           
//  - changed some if/else statements to switch/case                 
//==================================================================+

if(!class_exists('FPDF')) {
	define('FPDF_VERSION','1.52.1'); //2004-06-11 - changed by Nicola Asuni
	
	/**
	 * FPDF is a PHP class which allows to generate PDF files with straight PHP, 
	 * that is to say without using the PDFlib library.<br />
	 * This version contains changes by Nicola Asuni : [code style and formatting, php documentation (www.phpdoc.org), ISO page formats, minor improvements]
	 * @name FPDF
	 * @package FPDF
	 * @version 1.52.1
	 * @author Olivier PLATHEY 
	 * @license Freeware (You may use, modify and redistribute this software as you wish.)
	 */
	class FPDF {
		//Private properties
		
		/**
		 * @var current page number
		 */
		var $page;
		
		/**
		 * @var current object number
		 */
		var $n;
		
		/**
		 * @var array of object offsets
		 */
		var $offsets;
		
		/**
		 * @var buffer holding in-memory PDF
		 */
		var $buffer;
		
		/**
		 * @var array containing pages
		 */
		var $pages;
		
		/**
		 * @var current document state
		 */
		var $state;
		
		/**
		 * @var compression flag
		 */
		var $compress;
		
		/**
		 * @var default orientation
		 */
		var $DefOrientation;
		
		/**
		 * @var current orientation
		 */
		var $CurOrientation;
		
		/**
		 * @var array indicating orientation changes
		 */
		var $OrientationChanges;
		
		/**
		 * @var scale factor (number of points in user unit)
		 */
		var $k;
		
		/**
		 * @var width of page format in points
		 */
		var $fwPt;
		
		/**
		 * @var height of page format in points
		 */
		var $fhPt;
		
		/**
		 * @var width of page format in user unit
		 */
		var $fw;
		
		/**
		 * @var height of page format in user unit
		 */
		var $fh;
		
		/**
		 * @var current width of page in points
		 */
		var $wPt;
		
		/**
		 * @var current height of page in points
		 */
		var $hPt;
		
		/**
		 * @var current width of page in user unit
		 */
		var $w;
		
		/**
		 * @var current height of page in user unit
		 */
		var $h;
		
		/**
		 * @var left margin
		 */
		var $lMargin;
		
		/**
		 * @var top margin
		 */
		var $tMargin;
		
		/**
		 * @var right margin
		 */
		var $rMargin;
		
		/**
		 * @var page break margin
		 */
		var $bMargin;
		
		/**
		 * @var cell margin
		 */
		var $cMargin;
		
		/**
		 * @var current horizontal position in user unit for cell positioning
		 */
		var $x;
		
		/**
		 * @var current vertical position in user unit for cell positioning
		 */
		var $y;
		
		/**
		 * @var height of last cell printed
		 */
		var $lasth;
		
		/**
		 * @var line width in user unit
		 */
		var $LineWidth;
		
		/**
		 * @var array of standard font names
		 */
		var $CoreFonts;
		
		/**
		 * @var array of used fonts
		 */
		var $fonts;
		
		/**
		 * @var array of font files
		 */
		var $FontFiles;
		
		/**
		 * @var array of encoding differences
		 */
		var $diffs;
		
		/**
		 * @var array of used images
		 */
		var $images;
		
		/**
		 * @var array of links in pages
		 */
		var $PageLinks;
		
		/**
		 * @var array of internal links
		 */
		var $links;
		
		/**
		 * @var current font family
		 */
		var $FontFamily;
		
		/**
		 * @var current font style
		 */
		var $FontStyle;
		
		/**
		 * @var underlining flag
		 */
		var $underline;
		
		/**
		 * @var current font info
		 */
		var $CurrentFont;
		
		/**
		 * @var current font size in points
		 */
		var $FontSizePt;
		
		/**
		 * @var current font size in user unit
		 */
		var $FontSize;
		
		/**
		 * @var commands for drawing color
		 */
		var $DrawColor;
		
		/**
		 * @var commands for filling color
		 */
		var $FillColor;
		
		/**
		 * @var commands for text color
		 */
		var $TextColor;
		
		/**
		 * @var indicates whether fill and text colors are different
		 */
		var $ColorFlag;
		
		/**
		 * @var word spacing
		 */
		var $ws;
		
		/**
		 * @var automatic page breaking
		 */
		var $AutoPageBreak;
		
		/**
		 * @var threshold used to trigger page breaks
		 */
		var $PageBreakTrigger;
		
		/**
		 * @var flag set when processing footer
		 */
		var $InFooter;
		
		/**
		 * @var zoom display mode
		 */
		var $ZoomMode;
		
		/**
		 * @var layout display mode
		 */
		var $LayoutMode;
		
		/**
		 * @var title
		 */
		var $title;
		
		/**
		 * @var subject
		 */
		var $subject;
		
		/**
		 * @var author
		 */
		var $author;
		
		/**
		 * @var keywords
		 */
		var $keywords;
		
		/**
		 * @var creator
		 */
		var $creator;
		
		/**
		 * @var alias for total number of pages
		 */
		var $AliasNbPages;
		
		/**
		 * @var right-bottom corner X coordinate of inserted image
		 * @since 2002-07-31
		 * @author Nicola Asuni
		 */
		var $img_rb_x;       //
		
		/**
		 * @var right-bottom corner Y coordinate of inserted image
		 * @since 2002-07-31
		 * @author Nicola Asuni
		 */
		var $img_rb_y;
		
		//------------------------------------------------------------
		// Public methods 
		//------------------------------------------------------------
		
		/**
		 * This is the class constructor. 
		 * It allows to set up the page format, the orientation and 
		 * the measure unit used in all the methods (except for the font sizes).
		 * @since 1.0
		 * @param string $orientation page orientation. Possible values are (case insensitive):<ul><li>P or Portrait (default)</li><li>L or Landscape</li></ul>
		 * @param string $unit User measure unit. Possible values are:<ul><li>pt: point</li><li>mm: millimeter (default)</li><li>cm: centimeter</li><li>in: inch</li></ul><br />A point equals 1/72 of inch, that is to say about 0.35 mm (an inch being 2.54 cm). This is a very common unit in typography; font sizes are expressed in that unit.
		 * @param mixed $format The format used for pages. It can be either one of the following values (case insensitive) or a custom format in the form of a two-element array containing the width and the height (expressed in the unit given by unit).<ul><li>4A0</li><li>2A0</li><li>A0</li><li>A1</li><li>A2</li><li>A3</li><li>A4 (default)</li><li>A5</li><li>A6</li><li>A7</li><li>A8</li><li>A9</li><li>A10</li><li>B0</li><li>B1</li><li>B2</li><li>B3</li><li>B4</li><li>B5</li><li>B6</li><li>B7</li><li>B8</li><li>B9</li><li>B10</li><li>C0</li><li>C1</li><li>C2</li><li>C3</li><li>C4</li><li>C5</li><li>C6</li><li>C7</li><li>C8</li><li>C9</li><li>C10</li><li>RA0</li><li>RA1</li><li>RA2</li><li>RA3</li><li>RA4</li><li>SRA0</li><li>SRA1</li><li>SRA2</li><li>SRA3</li><li>SRA4</li><li>LETTER</li><li>LEGAL</li><li>EXECUTIVE</li><li>FOLIO</li></ul>
		 */
		function FPDF($orientation='P', $unit='mm', $format='A4') {
			//Some checks
			$this->_dochecks();
			//Initialization of properties
			$this->page=0;
			$this->n=2;
			$this->buffer='';
			$this->pages=array();
			$this->OrientationChanges=array();
			$this->state=0;
			$this->fonts=array();
			$this->FontFiles=array();
			$this->diffs=array();
			$this->images=array();
			$this->links=array();
			$this->InFooter=false;
			$this->lasth=0;
			$this->FontFamily='';
			$this->FontStyle='';
			$this->FontSizePt=12;
			$this->underline=false;
			$this->DrawColor='0 G';
			$this->FillColor='0 g';
			$this->TextColor='0 g';
			$this->ColorFlag=false;
			$this->ws=0;
			//Standard fonts
			$this->CoreFonts=array(
				'courier'=>'Courier',
				'courierB'=>'Courier-Bold',
				'courierI'=>'Courier-Oblique',
				'courierBI'=>'Courier-BoldOblique',
				'helvetica'=>'Helvetica',
				'helveticaB'=>'Helvetica-Bold',
				'helveticaI'=>'Helvetica-Oblique',
				'helveticaBI'=>'Helvetica-BoldOblique',
				'times'=>'Times-Roman',
				'timesB'=>'Times-Bold',
				'timesI'=>'Times-Italic',
				'timesBI'=>'Times-BoldItalic',
				'symbol'=>'Symbol',
				'zapfdingbats'=>'ZapfDingbats'
			);
			
			//Scale factor
			// 2003-06-11 - Nicola Asuni : changed if/else with switch statement
			switch (strtolower($unit)){
				case 'pt': {$this->k=1; break;}
				case 'mm': {$this->k=72/25.4; break;}
				case 'cm': {$this->k=72/2.54;; break;}
				case 'in': {$this->k=72;; break;}
				default : {$this->Error('Incorrect unit: '.$unit); break;}
			}
			
			//Page format
			if(is_string($format)) {
				// 2002-07-24 - Nicola Asuni
				// Added new page formats (45 standard ISO paper formats and 4 american common formats).
				// Paper cordinates are calculated in this way: (inches * 72) where (1 inch = 2.54 cm)
				switch (strtoupper($format)){
					case '4A0': {$format = array(4767.87,6740.79); break;}
					case '2A0': {$format = array(3370.39,4767.87); break;}
					case 'A0': {$format = array(2383.94,3370.39); break;}
					case 'A1': {$format = array(1683.78,2383.94); break;}
					case 'A2': {$format = array(1190.55,1683.78); break;}
					case 'A3': {$format = array(841.89,1190.55); break;}
					case 'A4': default: {$format = array(595.28,841.89); break;}
					case 'A5': {$format = array(419.53,595.28); break;}
					case 'A6': {$format = array(297.64,419.53); break;}
					case 'A7': {$format = array(209.76,297.64); break;}
					case 'A8': {$format = array(147.40,209.76); break;}
					case 'A9': {$format = array(104.88,147.40); break;}
					case 'A10': {$format = array(73.70,104.88); break;}
					case 'B0': {$format = array(2834.65,4008.19); break;}
					case 'B1': {$format = array(2004.09,2834.65); break;}
					case 'B2': {$format = array(1417.32,2004.09); break;}
					case 'B3': {$format = array(1000.63,1417.32); break;}
					case 'B4': {$format = array(708.66,1000.63); break;}
					case 'B5': {$format = array(498.90,708.66); break;}
					case 'B6': {$format = array(354.33,498.90); break;}
					case 'B7': {$format = array(249.45,354.33); break;}
					case 'B8': {$format = array(175.75,249.45); break;}
					case 'B9': {$format = array(124.72,175.75); break;}
					case 'B10': {$format = array(87.87,124.72); break;}
					case 'C0': {$format = array(2599.37,3676.54); break;}
					case 'C1': {$format = array(1836.85,2599.37); break;}
					case 'C2': {$format = array(1298.27,1836.85); break;}
					case 'C3': {$format = array(918.43,1298.27); break;}
					case 'C4': {$format = array(649.13,918.43); break;}
					case 'C5': {$format = array(459.21,649.13); break;}
					case 'C6': {$format = array(323.15,459.21); break;}
					case 'C7': {$format = array(229.61,323.15); break;}
					case 'C8': {$format = array(161.57,229.61); break;}
					case 'C9': {$format = array(113.39,161.57); break;}
					case 'C10': {$format = array(79.37,113.39); break;}
					case 'RA0': {$format = array(2437.80,3458.27); break;}
					case 'RA1': {$format = array(1729.13,2437.80); break;}
					case 'RA2': {$format = array(1218.90,1729.13); break;}
					case 'RA3': {$format = array(864.57,1218.90); break;}
					case 'RA4': {$format = array(609.45,864.57); break;}
					case 'SRA0': {$format = array(2551.18,3628.35); break;}
					case 'SRA1': {$format = array(1814.17,2551.18); break;}
					case 'SRA2': {$format = array(1275.59,1814.17); break;}
					case 'SRA3': {$format = array(907.09,1275.59); break;}
					case 'SRA4': {$format = array(637.80,907.09); break;}
					case 'LETTER': {$format = array(612.00,792.00); break;}
					case 'LEGAL': {$format = array(612.00,1008.00); break;}
					case 'EXECUTIVE': {$format = array(521.86,756.00); break;}
					case 'FOLIO': {$format = array(612.00,936.00); break;}
					// default: {$this->Error('Unknown page format: '.$format); break;}
					// END CHANGES Nicola Asuni
				}
				$this->fwPt=$format[0];
				$this->fhPt=$format[1];
			}
			else {
				$this->fwPt=$format[0]*$this->k;
				$this->fhPt=$format[1]*$this->k;
			}
			
			$this->fw=$this->fwPt/$this->k;
			$this->fh=$this->fhPt/$this->k;
			
			//Page orientation
			$orientation=strtolower($orientation);
			if($orientation=='p' or $orientation=='portrait') {
				$this->DefOrientation='P';
				$this->wPt=$this->fwPt;
				$this->hPt=$this->fhPt;
			}
			elseif($orientation=='l' or $orientation=='landscape') {
				$this->DefOrientation='L';
				$this->wPt=$this->fhPt;
				$this->hPt=$this->fwPt;
			}
			else {
				$this->Error('Incorrect orientation: '.$orientation);
			}
			
			$this->CurOrientation=$this->DefOrientation;
			$this->w=$this->wPt/$this->k;
			$this->h=$this->hPt/$this->k;
			//Page margins (1 cm)
			$margin=28.35/$this->k;
			$this->SetMargins($margin,$margin);
			//Interior cell margin (1 mm)
			$this->cMargin=$margin/10;
			//Line width (0.2 mm)
			$this->LineWidth=.567/$this->k;
			//Automatic page break
			$this->SetAutoPageBreak(true,2*$margin);
			//Full width display mode
			$this->SetDisplayMode('fullwidth');
			//Compression
			$this->SetCompression(true);
		}
		
		/**
		 * Defines the left, top and right margins. By default, they equal 1 cm. Call this method to change them.
		 * @param float $left Left margin.
		 * @param float $top Top margin. 
		 * @param float $right Right margin. Default value is the left one. 
		 * @since 1.0
		 * @see SetLeftMargin(), SetTopMargin(), SetRightMargin(), SetAutoPageBreak()
		 */
		function SetMargins($left, $top, $right=-1) {
			//Set left, top and right margins
			$this->lMargin=$left;
			$this->tMargin=$top;
			if($right==-1) {
				$right=$left;
			}
			$this->rMargin=$right;
		}
		
		/**
		 * Defines the left margin. The method can be called before creating the first page. If the current abscissa gets out of page, it is brought back to the margin. 
		 * @param float $margin The margin.
		 * @since 1.4
		 * @see SetTopMargin(), SetRightMargin(), SetAutoPageBreak(), SetMargins()
		 */
		function SetLeftMargin($margin) {
			//Set left margin
			$this->lMargin=$margin;
			if(($this->page>0) and ($this->x<$margin)) {
				$this->x=$margin;
			}
		}
		
		/**
		 * Defines the top margin. The method can be called before creating the first page.
		 * @param float $margin The margin.
		 * @since 1.5
		 * @see SetLeftMargin(), SetRightMargin(), SetAutoPageBreak(), SetMargins()
		 */
		function SetTopMargin($margin) {
			//Set top margin
			$this->tMargin=$margin;
		}
		
		/**
		 * Defines the right margin. The method can be called before creating the first page. 
		 * @param float $margin The margin.
		 * @since 1.5
		 * @see SetLeftMargin(), SetTopMargin(), SetAutoPageBreak(), SetMargins()
		 */
		function SetRightMargin($margin) {
			//Set right margin
			$this->rMargin=$margin;
		}
		
		/**
		 * Enables or disables the automatic page breaking mode. When enabling, the second parameter is the distance from the bottom of the page that defines the triggering limit. By default, the mode is on and the margin is 2 cm. 
		 * @param boolean $auto Boolean indicating if mode should be on or off. 
		 * @param float $margin Distance from the bottom of the page. 
		 * @since 1.0
		 * @see Cell(), MultiCell(), AcceptPageBreak()
		 */
		function SetAutoPageBreak($auto, $margin=0) {
			//Set auto page break mode and triggering margin
			$this->AutoPageBreak=$auto;
			$this->bMargin=$margin;
			$this->PageBreakTrigger=$this->h-$margin;
		}
		
		/**
		 * Defines the way the document is to be displayed by the viewer. The zoom level can be set: pages can be displayed entirely on screen, occupy the full width of the window, use real size, be scaled by a specific zooming factor or use viewer default (configured in the Preferences menu of Acrobat). The page layout can be specified too: single at once, continuous display, two columns or viewer default. By default, documents use the full width mode with continuous display. 
		 * @param mixed $zoom The zoom to use. It can be one of the following string values or a number indicating the zooming factor to use. <ul><li>fullpage: displays the entire page on screen </li><li>fullwidth: uses maximum width of window</li><li>real: uses real size (equivalent to 100% zoom)</li><li>default: uses viewer default mode</li></ul>
		 * @param string $layout The page layout. Possible values are:<ul><li>single: displays one page at once</li><li>continuous: displays pages continuously (default)</li><li>two: displays two pages on two columns</li><li>default: uses viewer default mode</li></ul>
		 * @since 1.2
		 */
		function SetDisplayMode($zoom, $layout='continuous') {
			//Set display mode in viewer
			if($zoom=='fullpage' or $zoom=='fullwidth' or $zoom=='real' or $zoom=='default' or !is_string($zoom)) {
				$this->ZoomMode=$zoom;
			}
			else {
				$this->Error('Incorrect zoom display mode: '.$zoom);
			}
			if($layout=='single' or $layout=='continuous' or $layout=='two' or $layout=='default') {
				$this->LayoutMode=$layout;
			}
			else {
				$this->Error('Incorrect layout display mode: '.$layout);
			}
		}
		
		/**
		 * Activates or deactivates page compression. When activated, the internal representation of each page is compressed, which leads to a compression ratio of about 2 for the resulting document. Compression is on by default.
		 * Note: the Zlib extension is required for this feature. If not present, compression will be turned off.
		 * @param boolean $compress Boolean indicating if compression must be enabled.
		 * @since 1.4
		 */
		function SetCompression($compress) {
			//Set page compression
			if(function_exists('gzcompress')) {
				$this->compress=$compress;
			}
			else {
				$this->compress=false;
			}
		}
		
		/**
		 * Defines the title of the document.
		 * @param string $title The title.
		 * @since 1.2
		 * @see SetAuthor(), SetCreator(), SetKeywords(), SetSubject()
		 */
		function SetTitle($title) {
			//Title of document
			$this->title=$title;
		}
		
		/**
		 * Defines the subject of the document.
		 * @param string $subject The subject.
		 * @since 1.2
		 * @see SetAuthor(), SetCreator(), SetKeywords(), SetTitle()
		 */
		function SetSubject($subject) {
			//Subject of document
			$this->subject=$subject;
		}
		
		/**
		 * Defines the author of the document.
		 * @param string $author The name of the author.
		 * @since 1.2
		 * @see SetCreator(), SetKeywords(), SetSubject(), SetTitle()
		 */
		function SetAuthor($author) {
			//Author of document
			$this->author=$author;
		}
		
		/**
		 * Associates keywords with the document, generally in the form 'keyword1 keyword2 ...'.
		 * @param string $keywords The list of keywords.
		 * @since 1.2
		 * @see SetAuthor(), SetCreator(), SetSubject(), SetTitle()
		 */
		function SetKeywords($keywords) {
			//Keywords of document
			$this->keywords=$keywords;
		}
		
		/**
		 * Defines the creator of the document. This is typically the name of the application that generates the PDF. 
		 * @param string $creator The name of the creator.
		 * @since 1.2
		 * @see SetAuthor(), SetKeywords(), SetSubject(), SetTitle()
		 */
		function SetCreator($creator) {
			//Creator of document
			$this->creator=$creator;
		}
		
		/**
		 * Defines an alias for the total number of pages. It will be substituted as the document is closed.<br />
		 * <b>Example:</b><br />
		 * <pre>
		 * class PDF extends FPDF {
		 * 	function Footer() {
		 * 		//Go to 1.5 cm from bottom
		 * 		$this->SetY(-15);
		 * 		//Select Arial italic 8
		 * 		$this->SetFont('Arial','I',8);
		 * 		//Print current and total page numbers
		 * 		$this->Cell(0,10,'Page '.$this->PageNo().'/{nb}',0,0,'C');
		 * 	}
		 * }
		 * $pdf=new PDF();
		 * $pdf->AliasNbPages(); 
		 * </pre>
		 * @param string $alias The alias. Default value: {nb}.
		 * @since 1.4
		 * @see PageNo(), Footer()
		 */
		function AliasNbPages($alias='{nb}') {
			//Define an alias for total number of pages
			$this->AliasNbPages=$alias;
		}
		
		/**
		 * This method is automatically called in case of fatal error; it simply outputs the message and halts the execution. An inherited class may override it to customize the error handling but should always halt the script, or the resulting document would probably be invalid.
		 * 2004-06-11 :: Nicola Asuni : changed bold tag with strong
		 * @param string $msg The error message
		 * @since 1.0
		 */
		function Error($msg) {
			//Fatal error
			die('<strong>FPDF error: </strong>'.$msg);
		}
		
		/**
		 * This method begins the generation of the PDF document. It is not necessary to call it explicitly because AddPage() does it automatically.
		 * Note: no page is created by this method
		 * @since 1.0
		 * @see AddPage(), Close()
		 */
		function Open() {
			//Begin document
			if($this->state==0) {
				$this->_begindoc();
			}
		}
		
		/**
		 * Terminates the PDF document. It is not necessary to call this method explicitly because Output() does it automatically. If the document contains no page, AddPage() is called to prevent from getting an invalid document.
		 * @since 1.0
		 * @see Open(), Output()
		 */
		function Close() {
			//Terminate document
			if($this->state==3) {
				return;
			}
			if($this->page==0) {
				$this->AddPage();
			}
			//Page footer
			$this->InFooter=true;
			$this->Footer();
			$this->InFooter=false;
			//Close page
			$this->_endpage();
			//Close document
			$this->_enddoc();
		}
		
		/**
		 * Adds a new page to the document. If a page is already present, the Footer() method is called first to output the footer. Then the page is added, the current position set to the top-left corner according to the left and top margins, and Header() is called to display the header. 
		 * The font which was set before calling is automatically restored. There is no need to call SetFont() again if you want to continue with the same font. The same is true for colors and line width. 
		 * The origin of the coordinate system is at the top-left corner and increasing ordinates go downwards. 
		 * @param string $orientation Page orientation. Possible values are (case insensitive):<ul><li>P or Portrait</li><li>L or Landscape</li></ul> The default value is the one passed to the constructor.
		 * @since 1.0
		 * @see FPDF(), Header(), Footer(), SetMargins()
		 */
		function AddPage($orientation='') {
			//Start a new page
			if($this->state==0) {
				$this->Open();
			}
			$family=$this->FontFamily;
			$style=$this->FontStyle.($this->underline ? 'U' : '');
			$size=$this->FontSizePt;
			$lw=$this->LineWidth;
			$dc=$this->DrawColor;
			$fc=$this->FillColor;
			$tc=$this->TextColor;
			$cf=$this->ColorFlag;
			if($this->page>0) {
				//Page footer
				$this->InFooter=true;
				$this->Footer();
				$this->InFooter=false;
				//Close page
				$this->_endpage();
			}
			//Start new page
			$this->_beginpage($orientation);
			//Set line cap style to square
			$this->_out('2 J');
			//Set line width
			$this->LineWidth=$lw;
			$this->_out(sprintf('%.2f w',$lw*$this->k));
			//Set font
			if($family) {
				$this->SetFont($family,$style,$size);
			}
			//Set colors
			$this->DrawColor=$dc;
			if($dc!='0 G') {
				$this->_out($dc);
			}
			$this->FillColor=$fc;
			if($fc!='0 g') {
				$this->_out($fc);
			}
			$this->TextColor=$tc;
			$this->ColorFlag=$cf;
			//Page header
			$this->Header();
			//Restore line width
			if($this->LineWidth!=$lw) {
				$this->LineWidth=$lw;
				$this->_out(sprintf('%.2f w',$lw*$this->k));
			}
			//Restore font
			if($family) {
				$this->SetFont($family,$style,$size);
			}
			//Restore colors
			if($this->DrawColor!=$dc) {
				$this->DrawColor=$dc;
				$this->_out($dc);
			}
			if($this->FillColor!=$fc) {
				$this->FillColor=$fc;
				$this->_out($fc);
			}
			$this->TextColor=$tc;
			$this->ColorFlag=$cf;
		}
		
		/**
		 * This method is used to render the page header. It is automatically called by AddPage() and should not be called directly by the application. The implementation in FPDF is empty, so you have to subclass it and override the method if you want a specific processing.<br />
		 * <b>Example:</b><br />
		 * <pre>
		 * class PDF extends FPDF {
		 * 	function Header() {
		 * 		//Select Arial bold 15
		 * 		$this->SetFont('Arial','B',15);
		 * 		//Move to the right
		 * 		$this->Cell(80);
		 * 		//Framed title
		 * 		$this->Cell(30,10,'Title',1,0,'C');
		 * 		//Line break
		 * 		$this->Ln(20);
		 * 	}
		 * }
		 * </pre>
		 * @since 1.0
		 * @see Footer()
		 */
		function Header() {
			//To be implemented in your own inherited class
		}
		
		/**
		 * This method is used to render the page footer. It is automatically called by AddPage() and Close() and should not be called directly by the application. The implementation in FPDF is empty, so you have to subclass it and override the method if you want a specific processing.
		 * <b>Example:</b><br />
		 * <pre>
		 * class PDF extends FPDF {
		 * 	function Footer() {
		 * 		//Go to 1.5 cm from bottom
		 * 		$this->SetY(-15);
		 * 		//Select Arial italic 8
		 * 		$this->SetFont('Arial','I',8);
		 * 		//Print centered page number
		 * 		$this->Cell(0,10,'Page '.$this->PageNo(),0,0,'C');
		 * 	}
		 * }
		 * </pre>
		 * @since 1.0
		 * @see Header()
		 */
		function Footer() {
			//To be implemented in your own inherited class
		}
		
		/**
		 * Returns the current page number. 
		 * @return int page number 
		 * @since 1.0
		 * @see AliasNbPages()
		 */
		function PageNo() {
			//Get current page number
			return $this->page;
		}
		
		/**
		 * Defines the color used for all drawing operations (lines, rectangles and cell borders). It can be expressed in RGB components or gray scale. The method can be called before the first page is created and the value is retained from page to page.
		 * @param int $r If g et b are given, red component; if not, indicates the gray level. Value between 0 and 255
		 * @param int $g Green component (between 0 and 255)
		 * @param int $b Blue component (between 0 and 255)
		 * @since 1.3
		 * @see SetFillColor(), SetTextColor(), Line(), Rect(), Cell(), MultiCell()
		 */
		function SetDrawColor($r, $g=-1, $b=-1) {
			//Set color for all stroking operations
			if(($r==0 and $g==0 and $b==0) or $g==-1) {
				$this->DrawColor=sprintf('%.3f G',$r/255);
			}
			else {
				$this->DrawColor=sprintf('%.3f %.3f %.3f RG',$r/255,$g/255,$b/255);
			}
			if($this->page>0) {
				$this->_out($this->DrawColor);
			}
		}
		
		/**
		 * Defines the color used for all filling operations (filled rectangles and cell backgrounds). It can be expressed in RGB components or gray scale. The method can be called before the first page is created and the value is retained from page to page.
		 * @param int $r If g et b are given, red component; if not, indicates the gray level. Value between 0 and 255
		 * @param int $g Green component (between 0 and 255)
		 * @param int $b Blue component (between 0 and 255)
		 * @since 1.3
		 * @see SetDrawColor(), SetTextColor(), Rect(), Cell(), MultiCell()
		 */
		function SetFillColor($r, $g=-1, $b=-1) {
			//Set color for all filling operations
			if(($r==0 and $g==0 and $b==0) or $g==-1) {
				$this->FillColor=sprintf('%.3f g',$r/255);
			}
			else {
				$this->FillColor=sprintf('%.3f %.3f %.3f rg',$r/255,$g/255,$b/255);
			}
			$this->ColorFlag=($this->FillColor!=$this->TextColor);
			if($this->page>0) {
				$this->_out($this->FillColor);
			}
		}
		
		/**
		 * Defines the color used for text. It can be expressed in RGB components or gray scale. The method can be called before the first page is created and the value is retained from page to page.
		 * @param int $r If g et b are given, red component; if not, indicates the gray level. Value between 0 and 255
		 * @param int $g Green component (between 0 and 255)
		 * @param int $b Blue component (between 0 and 255)
		 * @since 1.3
		 * @see SetDrawColor(), SetFillColor(), Text(), Cell(), MultiCell()
		 */
		function SetTextColor($r, $g=-1, $b=-1) {
			//Set color for text
			if(($r==0 and $g==0 and $b==0) or $g==-1) {
				$this->TextColor=sprintf('%.3f g',$r/255);
			}
			else {
				$this->TextColor=sprintf('%.3f %.3f %.3f rg',$r/255,$g/255,$b/255);
			}
			$this->ColorFlag=($this->FillColor!=$this->TextColor);
		}
		
		/**
		 * Returns the length of a string in user unit. A font must be selected. 
		 * @param string $s The string whose length is to be computed
		 * @return int
		 * @since 1.2
		 */
		function GetStringWidth($s) {
			//Get width of a string in the current font
			$s=(string)$s;
			$cw=&$this->CurrentFont['cw'];
			$w=0;
			$l=strlen($s);
			for($i=0;$i<$l;$i++) {
				$w+=$cw[$s{$i}];
			}
			return $w*$this->FontSize/1000;
		}
		
		/**
		 * Defines the line width. By default, the value equals 0.2 mm. The method can be called before the first page is created and the value is retained from page to page.
		 * @param float $width The width.
		 * @since 1.0
		 * @see Line(), Rect(), Cell(), MultiCell()
		 */
		function SetLineWidth($width) {
			//Set line width
			$this->LineWidth=$width;
			if($this->page>0) {
				$this->_out(sprintf('%.2f w',$width*$this->k));
			}
		}
		
		/**
		 * Draws a line between two points.
		 * @param float $x1 Abscissa of first point
		 * @param float $y1 Ordinate of first point
		 * @param float $x2 Abscissa of second point
		 * @param float $y2 Ordinate of second point
		 * @since 1.0
		 * @see SetLineWidth(), SetDrawColor()
		 */
		function Line($x1, $y1, $x2, $y2) {
			//Draw a line
			$this->_out(sprintf('%.2f %.2f m %.2f %.2f l S', $x1*$this->k, ($this->h-$y1)*$this->k, $x2*$this->k, ($this->h-$y2)*$this->k));
		}
		
		/**
		 * Outputs a rectangle. It can be drawn (border only), filled (with no border) or both.
		 * @param float $x Abscissa of upper-left corner
		 * @param float $y Ordinate of upper-left corner
		 * @param float $w Width
		 * @param float $h Height
		 * @param string $style Style of rendering. Possible values are:<ul><li>D or empty string: draw (default)</li><li>F: fill</li><li>DF or FD: draw and fill</li></ul>
		 * @since 1.0
		 * @see SetLineWidth(), SetDrawColor(), SetFillColor()
		 */
		function Rect($x, $y, $w, $h, $style='') {
			//Draw a rectangle
			if($style=='F') {
				$op='f';
			}
			elseif($style=='FD' or $style=='DF') {
				$op='B';
			}
			else {
				$op='S';
			}
			$this->_out(sprintf('%.2f %.2f %.2f %.2f re %s',$x*$this->k,($this->h-$y)*$this->k,$w*$this->k,-$h*$this->k,$op));
		}
		
		/**
		 * Imports a TrueType or Type1 font and makes it available. It is necessary to generate a font definition file first with the makefont.php utility. The definition file (and the font file itself when embedding) must be present either in the current directory or in the one indicated by FPDF_FONTPATH if the constant is defined. If it could not be found, the error "Could not include font definition file" is generated.
		 * <b>Example</b>:<br />
		 * <pre>
		 * $pdf->AddFont('Comic','I');  
		 * // is equivalent to: 
		 * $pdf->AddFont('Comic','I','comici.php'); 
		 * </pre>
		 * @param string $family Font family. The name can be chosen arbitrarily. If it is a standard family name, it will override the corresponding font.
		 * @param string $style Font style. Possible values are (case insensitive):<ul><li>empty string: regular (default)</li><li>B: bold</li><li>I: italic</li><li>BI or IB: bold italic</li></ul>
		 * @param string $file The font definition file. By default, the name is built from the family and style, in lower case with no space. 
		 * @since 1.5
		 * @see SetFont()
		 */
		function AddFont($family, $style='', $file='') {
			//Add a TrueType or Type1 font
			$family=strtolower($family);
			if($family=='arial') {
				$family='helvetica';
			}
			$style=strtoupper($style);
			if($style=='IB') {
				$style='BI';
			}
			if(isset($this->fonts[$family.$style])) {
				$this->Error('Font already added: '.$family.' '.$style);
			}
			if($file=='') {
				$file=str_replace(' ','',$family).strtolower($style).'.php';
			}
			if(defined('FPDF_FONTPATH')) {
				$file=FPDF_FONTPATH.$file;
			}
			include($file);
			if(!isset($name)) {
				$this->Error('Could not include font definition file');
			}
			$i=count($this->fonts)+1;
			$this->fonts[$family.$style]=array('i'=>$i,'type'=>$type,'name'=>$name,'desc'=>$desc,'up'=>$up,'ut'=>$ut,'cw'=>$cw,'enc'=>$enc,'file'=>$file);
			if($diff) {
				//Search existing encodings
				$d=0;
				$nb=count($this->diffs);
				for($i=1;$i<=$nb;$i++) {
					if($this->diffs[$i]==$diff) {
						$d=$i;
						break;
					}
				}
				if($d==0) {
					$d=$nb+1;
					$this->diffs[$d]=$diff;
				}
				$this->fonts[$family.$style]['diff']=$d;
			}
			if($file) {
				if($type=='TrueType') {
					$this->FontFiles[$file]=array('length1'=>$originalsize);
				}
				else {
					$this->FontFiles[$file]=array('length1'=>$size1,'length2'=>$size2);
				}
			}
		}
		
		/**
		 * Sets the font used to print character strings. It is mandatory to call this method at least once before printing text or the resulting document would not be valid. 
		 * The font can be either a standard one or a font added via the AddFont() method. Standard fonts use Windows encoding cp1252 (Western Europe). 
		 * The method can be called before the first page is created and the font is retained from page to page. 
	If you just wish to change the current font size, it is simpler to call SetFontSize(). 
		 * Note: for the standard fonts, the font metric files must be accessible. There are three possibilities for this:<ul><li>They are in the current directory (the one where the running script lies)</li><li>They are in one of the directories defined by the include_path parameter</li><li>They are in the directory defined by the FPDF_FONTPATH constant</li></ul><br />
		 * Example for the last case (note the trailing slash):<br />
		 * <pre>
		 * define('FPDF_FONTPATH','/home/www/font/');
		 * require('fpdf.php');
		 * 
		 * //Times regular 12
		 * $pdf->SetFont('Times');
		 * //Arial bold 14
		 * $pdf->SetFont('Arial','B',14);
		 * //Removes bold
		 * $pdf->SetFont('');
		 * //Times bold, italic and underlined 14
		 * $pdf->SetFont('Times','BIU'); 
		 * </pre><br />
		 * If the file corresponding to the requested font is not found, the error "Could not include font metric file" is generated. 
		 * @param string $family Family font. It can be either a name defined by AddFont() or one of the standard families (case insensitive):<ul><li>Courier (fixed-width)</li><li>Helvetica or Arial (synonymous; sans serif)</li><li>Times (serif)</li><li>Symbol (symbolic)</li><li>ZapfDingbats (symbolic)</li></ul>It is also possible to pass an empty string. In that case, the current family is retained. 
		 * @param string $style Font style. Possible values are (case insensitive):<ul><li>empty string: regular</li><li>B: bold</li><li>I: italic</li><li>U: underline</li></ul>or any combination. The default value is regular. Bold and italic styles do not apply to Symbol and ZapfDingbats
		 * @param float $size Font size in points. The default value is the current size. If no size has been specified since the beginning of the document, the value taken is 12
		 * @since 1.0
		 * @see AddFont(), SetFontSize(), Cell(), MultiCell(), Write()
		 */
		function SetFont($family, $style='', $size=0) {
			//Select a font; size given in points
			global $fpdf_charwidths;
			
			$family=strtolower($family);
			if($family=='') {
				$family=$this->FontFamily;
			}
			if($family=='arial') {
				$family='helvetica';
			}
			elseif($family=='symbol' or $family=='zapfdingbats') {
				$style='';
			}
			$style=strtoupper($style);
			if(is_int(strpos($style,'U'))) {
				$this->underline=true;
				$style=str_replace('U','',$style);
			}
			else {
				$this->underline=false;
			}
			if($style=='IB') {
				$style='BI';
			}
			if($size==0) {
				$size=$this->FontSizePt;
			}
			//Test if font is already selected
			if($this->FontFamily==$family and $this->FontStyle==$style and $this->FontSizePt==$size) {
				return;
			}
			//Test if used for the first time
			$fontkey=$family.$style;
			if(!isset($this->fonts[$fontkey])) {
				//Check if one of the standard fonts
				if(isset($this->CoreFonts[$fontkey])) {
					if(!isset($fpdf_charwidths[$fontkey])) {
						//Load metric file
						$file=$family;
						if($family=='times' or $family=='helvetica') {
							$file.=strtolower($style);
						}
						$file.='.php';
						if(defined('FPDF_FONTPATH')) {
							$file=FPDF_FONTPATH.$file;
						}
						include($file);
						if(!isset($fpdf_charwidths[$fontkey])) {
							$this->Error('Could not include font metric file');
						}
					}
					$i=count($this->fonts)+1;
					$this->fonts[$fontkey]=array('i'=>$i, 'type'=>'core', 'name'=>$this->CoreFonts[$fontkey], 'up'=>-100, 'ut'=>50, 'cw'=>$fpdf_charwidths[$fontkey]);
				}
				else {
					$this->Error('Undefined font: '.$family.' '.$style);
				}
			}
			//Select it
			$this->FontFamily=$family;
			$this->FontStyle=$style;
			$this->FontSizePt=$size;
			$this->FontSize=$size/$this->k;
			$this->CurrentFont=&$this->fonts[$fontkey];
			if($this->page>0) {
				$this->_out(sprintf('BT /F%d %.2f Tf ET', $this->CurrentFont['i'], $this->FontSizePt));
			}
		}
		
		/**
		 * Defines the size of the current font. 
		 * @param float $size The size (in points)
		 * @since 1.0
		 * @see SetFont()
		 */
		function SetFontSize($size) {
			//Set font size in points
			if($this->FontSizePt==$size) {
				return;
			}
			$this->FontSizePt=$size;
			$this->FontSize=$size/$this->k;
			if($this->page>0) {
				$this->_out(sprintf('BT /F%d %.2f Tf ET', $this->CurrentFont['i'], $this->FontSizePt));
			}
		}
		
		/**
		 * Creates a new internal link and returns its identifier. An internal link is a clickable area which directs to another place within the document.<br />
		 * The identifier can then be passed to Cell(), Write(), Image() or Link(). The destination is defined with SetLink(). 
		 * @since 1.5
		 * @see Cell(), Write(), Image(), Link(), SetLink()
		 */
		function AddLink() {
			//Create a new internal link
			$n=count($this->links)+1;
			$this->links[$n]=array(0,0);
			return $n;
		}
		
		/**
		 * Defines the page and position a link points to
		 * @param int $link The link identifier returned by AddLink()
		 * @param float $y Ordinate of target position; -1 indicates the current position. The default value is 0 (top of page)
		 * @param int $page Number of target page; -1 indicates the current page. This is the default value
		 * @since 1.5
		 * @see AddLink()
		 */
		function SetLink($link, $y=0, $page=-1) {
			//Set destination of internal link
			if($y==-1) {
				$y=$this->y;
			}
			if($page==-1) {
				$page=$this->page;
			}
			$this->links[$link]=array($page,$y);
		}
		
		/**
		 * Puts a link on a rectangular area of the page. Text or image links are generally put via Cell(), Write() or Image(), but this method can be useful for instance to define a clickable area inside an image.
		 * @param float $x Abscissa of the upper-left corner of the rectangle
		 * @param float $y Ordinate of the upper-left corner of the rectangle
		 * @param float $w Width of the rectangle
		 * @param float $h Height of the rectangle
		 * @param mixed $link URL or identifier returned by AddLink()
		 * @since 1.5
		 * @see AddLink(), Cell(), Write(), Image()
		 */
		function Link($x, $y, $w, $h, $link) {
			//Put a link on the page
			$this->PageLinks[$this->page][]=array($x*$this->k, $this->hPt-$y*$this->k, $w*$this->k, $h*$this->k,$link);
		}
		
		/**
		 * Prints a character string. The origin is on the left of the first charcter, on the baseline. This method allows to place a string precisely on the page, but it is usually easier to use Cell(), MultiCell() or Write() which are the standard methods to print text.
		 * @param float $x Abscissa of the origin
		 * @param float $y Ordinate of the origin
		 * @param string $txt String to print
		 * @since 1.0
		 * @see SetFont(), SetTextColor(), Cell(), MultiCell(), Write()
		 */
		function Text($x, $y, $txt) {
			//Output a string
			$s=sprintf('BT %.2f %.2f Td (%s) Tj ET',$x*$this->k, ($this->h-$y)*$this->k, $this->_escape($txt));
			if($this->underline and $txt!='') {
				$s.=' '.$this->_dounderline($x,$y,$txt);
			}
			if($this->ColorFlag) {
				$s='q '.$this->TextColor.' '.$s.' Q';
			}
			$this->_out($s);
		}
		
		/**
		 * Whenever a page break condition is met, the method is called, and the break is issued or not depending on the returned value. The default implementation returns a value according to the mode selected by SetAutoPageBreak().<br />
		 * This method is called automatically and should not be called directly by the application.<br />
		 * <b>Example:</b><br />
		 * The method is overriden in an inherited class in order to obtain a 3 column layout:<br />
		 * <pre>
		 * class PDF extends FPDF {
		 * 	var $col=0;
		 * 	
		 * 	function SetCol($col) {
		 * 		//Move position to a column
		 * 		$this->col=$col;
		 * 		$x=10+$col*65;
		 * 		$this->SetLeftMargin($x);
		 * 		$this->SetX($x);
		 * 	}
		 * 	
		 * 	function AcceptPageBreak() {
		 * 		if($this->col<2) {
		 * 			//Go to next column
		 * 			$this->SetCol($this->col+1);
		 * 			$this->SetY(10);
		 * 			return false;
		 * 		}
		 * 		else {
		 * 			//Go back to first column and issue page break
		 * 			$this->SetCol(0);
		 * 			return true;
		 * 		}
		 * 	}
		 * }
		 * 
		 * $pdf=new PDF();
		 * $pdf->Open();
		 * $pdf->AddPage();
		 * $pdf->SetFont('Arial','',12);
		 * for($i=1;$i<=300;$i++) {
		 *     $pdf->Cell(0,5,"Line $i",0,1);
		 * }
		 * $pdf->Output();  
		 * </pre>
		 * @return boolean
		 * @since 1.4
		 * @see SetAutoPageBreak()
		 */
		function AcceptPageBreak() {
			//Accept automatic page break or not
			return $this->AutoPageBreak;
		}
		
		/**
		 * Prints a cell (rectangular area) with optional borders, background color and character string. The upper-left corner of the cell corresponds to the current position. The text can be aligned or centered. After the call, the current position moves to the right or to the next line. It is possible to put a link on the text.<br />
		 * If automatic page breaking is enabled and the cell goes beyond the limit, a page break is done before outputting.
		 * @param float $w Cell width. If 0, the cell extends up to the right margin.
		 * @param float $h Cell height. Default value: 0.
		 * @param string $txt String to print. Default value: empty string.
		 * @param mixed $border Indicates if borders must be drawn around the cell. The value can be either a number:<ul><li>0: no border (default)</li><li>1: frame</li></ul>or a string containing some or all of the following characters (in any order):<ul><li>L: left</li><li>T: top</li><li>R: right</li><li>B: bottom</li></ul>
		 * @param int $ln Indicates where the current position should go after the call. Possible values are:<ul><li>0: to the right</li><li>1: to the beginning of the next line</li><li>2: below</li></ul>
	Putting 1 is equivalent to putting 0 and calling Ln() just after. Default value: 0.
		 * @param string $align Allows to center or align the text. Possible values are:<ul><li>L or empty string: left align (default value)</li><li>C: center</li><li>R: right align</li></ul>
		 * @param int $fill Indicates if the cell background must be painted (1) or transparent (0). Default value: 0.
		 * @param mixed $link URL or identifier returned by AddLink().
		 * @since 1.0
		 * @see SetFont(), SetDrawColor(), SetFillColor(), SetTextColor(), SetLineWidth(), AddLink(), Ln(), MultiCell(), Write(), SetAutoPageBreak()
		 */
		function Cell($w, $h=0, $txt='', $border=0, $ln=0, $align='', $fill=0, $link='') {
			//Output a cell
			$k=$this->k;
			if($this->y+$h>$this->PageBreakTrigger and !$this->InFooter and $this->AcceptPageBreak()) {
				//Automatic page break
				$x=$this->x;
				$ws=$this->ws;
				if($ws>0) {
					$this->ws=0;
					$this->_out('0 Tw');
				}
				$this->AddPage($this->CurOrientation);
				$this->x=$x;
				if($ws>0) {
					$this->ws=$ws;
					$this->_out(sprintf('%.3f Tw',$ws*$k));
				}
			}
			if($w==0) {
				$w=$this->w-$this->rMargin-$this->x;
			}
			$s='';
			if($fill==1 or $border==1) {
				if($fill==1) {
					$op=($border==1) ? 'B' : 'f';
				}
				else {
					$op='S';
				}
				$s=sprintf('%.2f %.2f %.2f %.2f re %s ', $this->x*$k, ($this->h-$this->y)*$k, $w*$k,-$h*$k,$op);
			}
			if(is_string($border)) {
				$x=$this->x;
				$y=$this->y;
				if(is_int(strpos($border,'L'))) {
					$s.=sprintf('%.2f %.2f m %.2f %.2f l S ',$x*$k,($this->h-$y)*$k,$x*$k,($this->h-($y+$h))*$k);
				}
				if(is_int(strpos($border,'T'))) {
					$s.=sprintf('%.2f %.2f m %.2f %.2f l S ',$x*$k,($this->h-$y)*$k,($x+$w)*$k,($this->h-$y)*$k);
				}
				if(is_int(strpos($border,'R'))) {
					$s.=sprintf('%.2f %.2f m %.2f %.2f l S ', ($x+$w)*$k, ($this->h-$y)*$k, ($x+$w)*$k, ($this->h-($y+$h))*$k);
				}
				if(is_int(strpos($border,'B'))) {
					$s.=sprintf('%.2f %.2f m %.2f %.2f l S ', $x*$k, ($this->h-($y+$h))*$k, ($x+$w)*$k, ($this->h-($y+$h))*$k);
				}
			}
			if($txt!='') {
				if($align=='R') {
					$dx=$w-$this->cMargin-$this->GetStringWidth($txt);
				}
				elseif($align=='C') {
					$dx=($w-$this->GetStringWidth($txt))/2;
				}
				else {
					$dx=$this->cMargin;
				}
				if($this->ColorFlag) {
					$s.='q '.$this->TextColor.' ';
				}
				$txt2=str_replace(')','\\)',str_replace('(','\\(',str_replace('\\','\\\\',$txt)));
				$s.=sprintf('BT %.2f %.2f Td (%s) Tj ET', ($this->x+$dx)*$k, ($this->h-($this->y+.5*$h+.3*$this->FontSize))*$k, $txt2);
				if($this->underline) {
					$s.=' '.$this->_dounderline($this->x+$dx,$this->y+.5*$h+.3*$this->FontSize,$txt);
				}
				if($this->ColorFlag) {
					$s.=' Q';
				}
				if($link) {
					$this->Link($this->x+$dx,$this->y+.5*$h-.5*$this->FontSize, $this->GetStringWidth($txt), $this->FontSize,$link);
				}
			}
			if($s) {
				$this->_out($s);
			}
			$this->lasth=$h;
			if($ln>0) {
				//Go to next line
				$this->y+=$h;
				if($ln==1) {
					$this->x=$this->lMargin;
				}
			}
			else {
				$this->x+=$w;
			}
		}
		
		/**
		 * This method allows printing text with line breaks. They can be automatic (as soon as the text reaches the right border of the cell) or explicit (via the \n character). As many cells as necessary are output, one below the other.<br />
		 * Text can be aligned, centered or justified. The cell block can be framed and the background painted. 
		 * @param float $w Width of cells. If 0, they extend up to the right margin of the page.
		 * @param float $h Height of cells.
		 * @param string $txt String to print
		 * @param mixed $border Indicates if borders must be drawn around the cell block. The value can be either a number:<ul><li>0: no border (default)</li><li>1: frame</li></ul>or a string containing some or all of the following characters (in any order):<ul><li>L: left</li><li>T: top</li><li>R: right</li><li>B: bottom</li></ul>
		 * @param int $ln Indicates where the current position should go after the call. Possible values are:<ul><li>0: to the right</li><li>1: to the beginning of the next line</li><li>2: below</li></ul>
	Putting 1 is equivalent to putting 0 and calling Ln() just after. Default value: 0.
		 * @param string $align Allows to center or align the text. Possible values are:<ul><li>L or empty string: left align</li><li>C: center</li><li>R: right align</li><li>J: justification (default value)</li></ul>
		 * @param int $fill Indicates if the cell background must be painted (1) or transparent (0). Default value: 0.
		 * @since 1.3
		 * @see SetFont(), SetDrawColor(), SetFillColor(), SetTextColor(), SetLineWidth(), Cell(), Write(), SetAutoPageBreak()
		 */
		function MultiCell($w, $h, $txt, $border=0, $align='J', $fill=0) {
			//Output text with automatic or explicit line breaks
			$cw=&$this->CurrentFont['cw'];
			if($w==0) {
				$w=$this->w-$this->rMargin-$this->x;
			}
			$wmax=($w-2*$this->cMargin)*1000/$this->FontSize;
			$s=str_replace("\r",'',$txt);
			$nb=strlen($s);
			if($nb>0 and $s[$nb-1]=="\n") {
				$nb--;
			}
			$b=0;
			if($border) {
				if($border==1) {
					$border='LTRB';
					$b='LRT';
					$b2='LR';
				}
				else {
					$b2='';
					if(is_int(strpos($border,'L'))) {
						$b2.='L';
					}
					if(is_int(strpos($border,'R'))) {
						$b2.='R';
					}
					$b=is_int(strpos($border,'T')) ? $b2.'T' : $b2;
				}
			}
			$sep=-1;
			$i=0;
			$j=0;
			$l=0;
			$ns=0;
			$nl=1;
			while($i<$nb) {
				//Get next character
				$c=$s{$i};
				if($c=="\n") {
					//Explicit line break
					if($this->ws>0) {
						$this->ws=0;
						$this->_out('0 Tw');
					}
					$this->Cell($w,$h,substr($s, $j, $i-$j), $b,2, $align, $fill);
					$i++;
					$sep=-1;
					$j=$i;
					$l=0;
					$ns=0;
					$nl++;
					if($border and $nl==2) {
						$b=$b2;
					}
					continue;
				}
				if($c==' ') {
					$sep=$i;
					$ls=$l;
					$ns++;
				}
				$l+=$cw[$c];
				if($l>$wmax) {
					//Automatic line break
					if($sep==-1) {
						if($i==$j) {
							$i++;
						}
						if($this->ws>0) {
							$this->ws=0;
							$this->_out('0 Tw');
						}
						$this->Cell($w,$h,substr($s,$j,$i-$j),$b,2,$align,$fill);
					}
					else {
						if($align=='J') {
							$this->ws=($ns>1) ? ($wmax-$ls)/1000*$this->FontSize/($ns-1) : 0;
							$this->_out(sprintf('%.3f Tw',$this->ws*$this->k));
						}
						$this->Cell($w,$h,substr($s,$j,$sep-$j),$b,2,$align,$fill);
						$i=$sep+1;
					}
					$sep=-1;
					$j=$i;
					$l=0;
					$ns=0;
					$nl++;
					if($border and $nl==2) {
						$b=$b2;
					}
				}
				else {
					$i++;
				}
			}
			//Last chunk
			if($this->ws>0) {
				$this->ws=0;
				$this->_out('0 Tw');
			}
			if($border and is_int(strpos($border,'B'))) {
				$b.='B';
			}
			$this->Cell($w,$h,substr($s,$j,$i-$j),$b,2,$align,$fill);
			$this->x=$this->lMargin;
		}
		
		/**
		 * This method prints text from the current position. When the right margin is reached (or the \n character is met) a line break occurs and text continues from the left margin. Upon method exit, the current position is left just at the end of the text. It is possible to put a link on the text.<br />
		 * <b>Example:</b><br />
		 * <pre>
		 * //Begin with regular font
		 * $pdf->SetFont('Arial','',14);
		 * $pdf->Write(5,'Visit ');
		 * //Then put a blue underlined link
		 * $pdf->SetTextColor(0,0,255);
		 * $pdf->SetFont('','U');
		 * $pdf->Write(5,'www.fpdf.org','http://www.fpdf.org');  
		 * </pre>
		 * @param float $h Line height
		 * @param string $txt String to print
		 * @param mixed $link URL or identifier returned by AddLink()
		 * @since 1.5
		 * @see SetFont(), SetTextColor(), AddLink(), MultiCell(), SetAutoPageBreak()
		 */
		function Write($h, $txt, $link='') {
			//Output text in flowing mode
			$cw=&$this->CurrentFont['cw'];
			$w=$this->w-$this->rMargin-$this->x;
			$wmax=($w-2*$this->cMargin)*1000/$this->FontSize;
			$s=str_replace("\r",'',$txt);
			$nb=strlen($s);
			$sep=-1;
			$i=0;
			$j=0;
			$l=0;
			$nl=1;
			while($i<$nb) {
				//Get next character
				$c=$s{$i};
				if($c=="\n") {
					//Explicit line break
					$this->Cell($w,$h,substr($s,$j, $i-$j), 0, 2, '', 0, $link);
					$i++;
					$sep=-1;
					$j=$i;
					$l=0;
					if($nl==1) {
						$this->x=$this->lMargin;
						$w=$this->w-$this->rMargin-$this->x;
						$wmax=($w-2*$this->cMargin)*1000/$this->FontSize;
					}
					$nl++;
					continue;
				}
				if($c==' ') {
					$sep=$i;
				}
				$l+=$cw[$c];
				if($l>$wmax) {
					//Automatic line break
					if($sep==-1) {
						if($this->x>$this->lMargin) {
							//Move to next line
							$this->x=$this->lMargin;
							$this->y+=$h;
							$w=$this->w-$this->rMargin-$this->x;
							$wmax=($w-2*$this->cMargin)*1000/$this->FontSize;
							$i++;
							$nl++;
							continue;
						}
						if($i==$j) {
							$i++;
						}
						$this->Cell($w, $h, substr($s,$j,$i-$j), 0, 2, '', 0, $link);
					}
					else {
						$this->Cell($w, $h, substr($s,$j,$sep-$j), 0, 2, '', 0, $link);
						$i=$sep+1;
					}
					$sep=-1;
					$j=$i;
					$l=0;
					if($nl==1) {
						$this->x=$this->lMargin;
						$w=$this->w-$this->rMargin-$this->x;
						$wmax=($w-2*$this->cMargin)*1000/$this->FontSize;
					}
					$nl++;
				}
				else {
					$i++;
				}
			}
			//Last chunk
			if($i!=$j) {
				$this->Cell($l/1000*$this->FontSize, $h, substr($s,$j), 0, 0, '', 0, $link);
			}
		}
		
		/**
		 * Puts an image in the page. The upper-left corner must be given. The dimensions can be specified in different ways:<ul><li>explicit width and height (expressed in user unit)</li><li>one explicit dimension, the other being calculated automatically in order to keep the original proportions</li><li>no explicit dimension, in which case the image is put at 72 dpi</li></ul>
		 * Supported formats are JPEG and PNG. 
		 * For JPEG, all flavors are allowed:<ul><li>gray scales</li><li>true colors (24 bits)</li><li>CMYK (32 bits)</li></ul>
		 * For PNG, are allowed:<ul><li>gray scales on at most 8 bits (256 levels)</li><li>indexed colors</li><li>true colors (24 bits)</li></ul>
		 * but are not supported:<ul><li>Interlacing</li><li>Alpha channel</li></ul>
		 * If a transparent color is defined, it will be taken into account (but will be only interpreted by Acrobat 4 and above).<br />
		 * The format can be specified explicitly or inferred from the file extension.<br />
		 * It is possible to put a link on the image.<br />
		 * Remark: if an image is used several times, only one copy will be embedded in the file.<br />
		 * @param string $file Name of the file containing the image. 
		 * @param float $x Abscissa of the upper-left corner. 
		 * @param float $y Ordinate of the upper-left corner. 
		 * @param float $w Width of the image in the page. If not specified or equal to zero, it is automatically calculated. 
		 * @param float $h Height of the image in the page. If not specified or equal to zero, it is automatically calculated. 
		 * @param string $type Image format. Possible values are (case insensitive): JPG, JPEG, PNG. If not specified, the type is inferred from the file extension. 
		 * @param mixed $link URL or identifier returned by AddLink(). 
		 * @since 1.1
		 * @see AddLink()
		 */
		function Image($file, $x, $y, $w=0, $h=0, $type='', $link='') {
			//Put an image on the page
			if(!isset($this->images[$file])) {
				//First use of image, get info
				if($type=='') {
					$pos=strrpos($file,'.');
					if(!$pos) {
						$this->Error('Image file has no extension and no type was specified: '.$file);
					}
					$type=substr($file,$pos+1);
				}
				$type=strtolower($type);
				$mqr=get_magic_quotes_runtime();
				set_magic_quotes_runtime(0);
				if($type=='jpg' or $type=='jpeg') {
					$info=$this->_parsejpg($file);
				}
				elseif($type=='png') {
					$info=$this->_parsepng($file);
				}
				else {
					//Allow for additional formats
					$mtd='_parse'.$type;
					if(!method_exists($this,$mtd)) {
						$this->Error('Unsupported image type: '.$type);
					}
					$info=$this->$mtd($file);
				}
				set_magic_quotes_runtime($mqr);
				$info['i']=count($this->images)+1;
				$this->images[$file]=$info;
			}
			else {
				$info=$this->images[$file];
			}
			//Automatic width and height calculation if needed
			if($w==0 and $h==0) {
				//Put image at 72 dpi
				$w=$info['w']/$this->k;
				$h=$info['h']/$this->k;
			}
			if($w==0) {
				$w=$h*$info['w']/$info['h'];
			}
			if($h==0) {
				$h=$w*$info['h']/$info['w'];
			}
			$this->_out(sprintf('q %.2f 0 0 %.2f %.2f %.2f cm /I%d Do Q',$w*$this->k,$h*$this->k,$x*$this->k,($this->h-($y+$h))*$this->k,$info['i']));
			if($link) {
				$this->Link($x,$y,$w,$h,$link);
			}
			
			//2002-07-31 - Nicola Asuni
			// set right-bottom corner coordinates
			$this->img_rb_x = $x + $w;
			$this->img_rb_y = $y + $h;
		}
		
		/**
		 * Performs a line break. The current abscissa goes back to the left margin and the ordinate increases by the amount passed in parameter.
		 * @param float $h The height of the break. By default, the value equals the height of the last printed cell.
		 * @since 1.0
		 * @see Cell()
		 */
		function Ln($h='') {
			//Line feed; default value is last cell height
			$this->x=$this->lMargin;
			if(is_string($h)) {
				$this->y+=$this->lasth;
			}
			else {
				$this->y+=$h;
			}
		}
		
		/**
		 * Returns the abscissa of the current position. 
		 * @return float
		 * @since 1.2
		 * @see SetX(), GetY(), SetY()
		 */
		function GetX() {
			//Get x position
			return $this->x;
		}
		
		/**
		 * Defines the abscissa of the current position. If the passed value is negative, it is relative to the right of the page.
		 * @param float $x The value of the abscissa.
		 * @since 1.2
		 * @see GetX(), GetY(), SetY(), SetXY()
		 */
		function SetX($x) {
			//Set x position
			if($x>=0) {
				$this->x=$x;
			}
			else {
				$this->x=$this->w+$x;
			}
		}
		
		/**
		 * Returns the ordinate of the current position.
		 * @return float
		 * @since 1.0
		 * @see SetY(), GetX(), SetX()
		 */
		function GetY() {
			//Get y position
			return $this->y;
		}
		
		/**
		 * Moves the current abscissa back to the left margin and sets the ordinate. If the passed value is negative, it is relative to the bottom of the page.
		 * @param float $y The value of the ordinate. 
		 * @since 1.0
		 * @see GetX(), GetY(), SetY(), SetXY()
		 */
		function SetY($y) {
			//Set y position and reset x
			$this->x=$this->lMargin;
			if($y>=0) {
				$this->y=$y;
			}
			else {
				$this->y=$this->h+$y;
			}
		}
		
		/**
		 * Defines the abscissa and ordinate of the current position. If the passed values are negative, they are relative respectively to the right and bottom of the page. 
		 * @param float $x The value of the abscissa
		 * @param float $y The value of the ordinate
		 * @since 1.2
		 * @see SetX(), SetY()
		 */
		function SetXY($x, $y) {
			//Set x and y positions
			$this->SetY($y);
			$this->SetX($x);
		}
		
		/**
		 * Send the document to a given destination: string, local file or browser. In the last case, the plug-in may be used (if present) or a download ("Save as" dialog box) may be forced.<br />
		 * The method first calls Close() if necessary to terminate the document.
		 * @param string $name The name of the file. If not given, the document will be sent to the browser (destination I) with the name doc.pdf.
		 * @param string $dest Destination where to send the document. It can take one of the following values:<ul><li>I: send the file inline to the browser. The plug-in is used if available. The name given by name is used when one selects the "Save as" option on the link generating the PDF.</li><li>D: send to the browser and force a file download with the name given by name.</li><li>F: save to a local file with the name given by name.</li><li>S: return the document as a string. name is ignored.</li></ul>If the parameter is not specified but a name is given, destination is F. If no parameter is specified at all, destination is I.<br />Note: for compatibility with previous versions, a boolean value is also accepted (false for F and true for D). 
		 * @since 1.0
		 * @see Close()
		 */
		function Output($name='',$dest='') {
			//Output PDF to some destination
			global $HTTP_SERVER_VARS;
		
			//Finish document if necessary
			if($this->state<3) {
				$this->Close();
			}
			//Normalize parameters
			if(is_bool($dest)) {
				$dest=$dest ? 'D' : 'F';
			}
			$dest=strtoupper($dest);
			if($dest=='') {
				if($name=='') {
					$name='doc.pdf';
					$dest='I';
				}
				else {
					$dest='F';
				}
			}
			switch($dest) {
				case 'I': {
					//Send to standard output
					if(isset($HTTP_SERVER_VARS['SERVER_NAME'])) {
						//We send to a browser
						Header('Content-Type: application/pdf');
						if(headers_sent()) {
							$this->Error('Some data has already been output to browser, can\'t send PDF file');
						}
						Header('Content-Length: '.strlen($this->buffer));
						Header('Content-disposition: inline; filename='.$name);
					}
					echo $this->buffer;
					break;
				}
				case 'D': {
					//Download file
					if(isset($HTTP_SERVER_VARS['HTTP_USER_AGENT']) and strpos($HTTP_SERVER_VARS['HTTP_USER_AGENT'],'MSIE')) {
						Header('Content-Type: application/force-download');
					}
					else {
						Header('Content-Type: application/octet-stream');
					}
					if(headers_sent()) {
						$this->Error('Some data has already been output to browser, can\'t send PDF file');
					}
					Header('Content-Length: '.strlen($this->buffer));
					Header('Content-disposition: attachment; filename='.$name);
					echo $this->buffer;
					break;
				}
				case 'F': {
					//Save to local file
					$f=fopen($name,'wb');
					if(!$f) {
						$this->Error('Unable to create output file: '.$name);
					}
					fwrite($f,$this->buffer,strlen($this->buffer));
					fclose($f);
					break;
				}
				case 'S': {
					//Return as a string
					return $this->buffer;
				}
				default: {
					$this->Error('Incorrect output destination: '.$dest);
				}
			}
			return '';
		}
		
		/*******************************************************************************
		*                                                                              *
		*                              Protected methods                               *
		*                                                                              *
		*******************************************************************************/
		/**
		 * Check for locale-related bug
		 * @access private
		 */
		function _dochecks() {
			//Check for locale-related bug
			if(1.1==1) {
				$this->Error('Don\'t alter the locale before including class file');
			}
			//Check for decimal separator
			if(sprintf('%.1f',1.0)!='1.0') {
				setlocale(LC_NUMERIC,'C');
			}
		}
		
		/**
		 * Start document
		 * @access private
		 */
		function _begindoc() {
			//Start document
			$this->state=1;
			$this->_out('%PDF-1.3');
		}
		
		/**
		 * _putpages
		 * @access private
		 */
		function _putpages() {
			$nb=$this->page;
			if(!empty($this->AliasNbPages)) {
				//Replace number of pages
				for($n=1;$n<=$nb;$n++) {
					$this->pages[$n]=str_replace($this->AliasNbPages,$nb,$this->pages[$n]);
				}
			}
			if($this->DefOrientation=='P') {
				$wPt=$this->fwPt;
				$hPt=$this->fhPt;
			}
			else {
				$wPt=$this->fhPt;
				$hPt=$this->fwPt;
			}
			$filter=($this->compress) ? '/Filter /FlateDecode ' : '';
			for($n=1;$n<=$nb;$n++) {
				//Page
				$this->_newobj();
				$this->_out('<</Type /Page');
				$this->_out('/Parent 1 0 R');
				if(isset($this->OrientationChanges[$n])) {
					$this->_out(sprintf('/MediaBox [0 0 %.2f %.2f]',$hPt,$wPt));
				}
				$this->_out('/Resources 2 0 R');
				if(isset($this->PageLinks[$n])) {
					//Links
					$annots='/Annots [';
					foreach($this->PageLinks[$n] as $pl) {
						$rect=sprintf('%.2f %.2f %.2f %.2f',$pl[0],$pl[1],$pl[0]+$pl[2],$pl[1]-$pl[3]);
						$annots.='<</Type /Annot /Subtype /Link /Rect ['.$rect.'] /Border [0 0 0] ';
						if(is_string($pl[4])) {
							$annots.='/A <</S /URI /URI '.$this->_textstring($pl[4]).'>>>>';
						}
						else {
							$l=$this->links[$pl[4]];
							$h=isset($this->OrientationChanges[$l[0]]) ? $wPt : $hPt;
							$annots.=sprintf('/Dest [%d 0 R /XYZ 0 %.2f null]>>',1+2*$l[0],$h-$l[1]*$this->k);
						}
					}
					$this->_out($annots.']');
				}
				$this->_out('/Contents '.($this->n+1).' 0 R>>');
				$this->_out('endobj');
				//Page content
				$p=($this->compress) ? gzcompress($this->pages[$n]) : $this->pages[$n];
				$this->_newobj();
				$this->_out('<<'.$filter.'/Length '.strlen($p).'>>');
				$this->_putstream($p);
				$this->_out('endobj');
			}
			//Pages root
			$this->offsets[1]=strlen($this->buffer);
			$this->_out('1 0 obj');
			$this->_out('<</Type /Pages');
			$kids='/Kids [';
			for($i=0;$i<$nb;$i++) {
				$kids.=(3+2*$i).' 0 R ';
			}
			$this->_out($kids.']');
			$this->_out('/Count '.$nb);
			$this->_out(sprintf('/MediaBox [0 0 %.2f %.2f]',$wPt,$hPt));
			$this->_out('>>');
			$this->_out('endobj');
		}
		
		/**
		 * _putfonts
		 * @access private
		 */
		function _putfonts() {
			$nf=$this->n;
			foreach($this->diffs as $diff) {
				//Encodings
				$this->_newobj();
				$this->_out('<</Type /Encoding /BaseEncoding /WinAnsiEncoding /Differences ['.$diff.']>>');
				$this->_out('endobj');
			}
			$mqr=get_magic_quotes_runtime();
			set_magic_quotes_runtime(0);
			foreach($this->FontFiles as $file=>$info) {
				//Font file embedding
				$this->_newobj();
				$this->FontFiles[$file]['n']=$this->n;
				if(defined('FPDF_FONTPATH')) {
					$file=FPDF_FONTPATH.$file;
				}
				$size=filesize($file);
				if(!$size) {
					$this->Error('Font file not found');
				}
				$this->_out('<</Length '.$size);
				if(substr($file,-2)=='.z') {
					$this->_out('/Filter /FlateDecode');
				}
				$this->_out('/Length1 '.$info['length1']);
				if(isset($info['length2'])) {
					$this->_out('/Length2 '.$info['length2'].' /Length3 0');
				}
				$this->_out('>>');
				$f=fopen($file,'rb');
				$this->_putstream(fread($f,$size));
				fclose($f);
				$this->_out('endobj');
			}
			set_magic_quotes_runtime($mqr);
			foreach($this->fonts as $k=>$font) {
				//Font objects
				$this->fonts[$k]['n']=$this->n+1;
				$type=$font['type'];
				$name=$font['name'];
				if($type=='core') {
					//Standard font
					$this->_newobj();
					$this->_out('<</Type /Font');
					$this->_out('/BaseFont /'.$name);
					$this->_out('/Subtype /Type1');
					if($name!='Symbol' and $name!='ZapfDingbats') {
						$this->_out('/Encoding /WinAnsiEncoding');
					}
					$this->_out('>>');
					$this->_out('endobj');
				}
				elseif($type=='Type1' or $type=='TrueType') {
					//Additional Type1 or TrueType font
					$this->_newobj();
					$this->_out('<</Type /Font');
					$this->_out('/BaseFont /'.$name);
					$this->_out('/Subtype /'.$type);
					$this->_out('/FirstChar 32 /LastChar 255');
					$this->_out('/Widths '.($this->n+1).' 0 R');
					$this->_out('/FontDescriptor '.($this->n+2).' 0 R');
					if($font['enc']) {
						if(isset($font['diff'])) {
							$this->_out('/Encoding '.($nf+$font['diff']).' 0 R');
						}
						else {
							$this->_out('/Encoding /WinAnsiEncoding');
						}
					}
					$this->_out('>>');
					$this->_out('endobj');
					//Widths
					$this->_newobj();
					$cw=&$font['cw'];
					$s='[';
					for($i=32;$i<=255;$i++) {
						$s.=$cw[chr($i)].' ';
					}
					$this->_out($s.']');
					$this->_out('endobj');
					//Descriptor
					$this->_newobj();
					$s='<</Type /FontDescriptor /FontName /'.$name;
					foreach($font['desc'] as $k=>$v) {
						$s.=' /'.$k.' '.$v;
					}
					$file=$font['file'];
					if($file) {
						$s.=' /FontFile'.($type=='Type1' ? '' : '2').' '.$this->FontFiles[$file]['n'].' 0 R';
					}
					$this->_out($s.'>>');
					$this->_out('endobj');
				}
				else {
					//Allow for additional types
					$mtd='_put'.strtolower($type);
					if(!method_exists($this,$mtd)) {
						$this->Error('Unsupported font type: '.$type);
					}
					$this->$mtd($font);
				}
			}
		}
		
		/**
		 * _putimages
		 * @access private
		 */
		function _putimages() {
			$filter=($this->compress) ? '/Filter /FlateDecode ' : '';
			reset($this->images);
			while(list($file,$info)=each($this->images)) {
				$this->_newobj();
				$this->images[$file]['n']=$this->n;
				$this->_out('<</Type /XObject');
				$this->_out('/Subtype /Image');
				$this->_out('/Width '.$info['w']);
				$this->_out('/Height '.$info['h']);
				if($info['cs']=='Indexed') {
					$this->_out('/ColorSpace [/Indexed /DeviceRGB '.(strlen($info['pal'])/3-1).' '.($this->n+1).' 0 R]');
				}
				else {
					$this->_out('/ColorSpace /'.$info['cs']);
					if($info['cs']=='DeviceCMYK') {
						$this->_out('/Decode [1 0 1 0 1 0 1 0]');
					}
				}
				$this->_out('/BitsPerComponent '.$info['bpc']);
				$this->_out('/Filter /'.$info['f']);
				if(isset($info['parms'])) {
					$this->_out($info['parms']);
				}
				if(isset($info['trns']) and is_array($info['trns'])) {
					$trns='';
					for($i=0;$i<count($info['trns']);$i++) {
						$trns.=$info['trns'][$i].' '.$info['trns'][$i].' ';
					}
					$this->_out('/Mask ['.$trns.']');
				}
				$this->_out('/Length '.strlen($info['data']).'>>');
				$this->_putstream($info['data']);
				unset($this->images[$file]['data']);
				$this->_out('endobj');
				//Palette
				if($info['cs']=='Indexed') {
					$this->_newobj();
					$pal=($this->compress) ? gzcompress($info['pal']) : $info['pal'];
					$this->_out('<<'.$filter.'/Length '.strlen($pal).'>>');
					$this->_putstream($pal);
					$this->_out('endobj');
				}
			}
		}
		
		/**
		 * _putresources
		 * @access private
		 */
		function _putresources() {
			$this->_putfonts();
			$this->_putimages();
			//Resource dictionary
			$this->offsets[2]=strlen($this->buffer);
			$this->_out('2 0 obj');
			$this->_out('<</ProcSet [/PDF /Text /ImageB /ImageC /ImageI]');
			$this->_out('/Font <<');
			foreach($this->fonts as $font) {
				$this->_out('/F'.$font['i'].' '.$font['n'].' 0 R');
			}
			$this->_out('>>');
			if(count($this->images)) {
				$this->_out('/XObject <<');
				foreach($this->images as $image) {
					$this->_out('/I'.$image['i'].' '.$image['n'].' 0 R');
				}
				$this->_out('>>');
			}
			$this->_out('>>');
			$this->_out('endobj');
		}
		
		/**
		 * _putinfo
		 * @access private
		 */
		function _putinfo() {
			$this->_out('/Producer '.$this->_textstring('FPDF '.FPDF_VERSION));
			if(!empty($this->title)) {
				$this->_out('/Title '.$this->_textstring($this->title));
			}
			if(!empty($this->subject)) {
				$this->_out('/Subject '.$this->_textstring($this->subject));
			}
			if(!empty($this->author)) {
				$this->_out('/Author '.$this->_textstring($this->author));
			}
			if(!empty($this->keywords)) {
				$this->_out('/Keywords '.$this->_textstring($this->keywords));
			}
			if(!empty($this->creator)) {
				$this->_out('/Creator '.$this->_textstring($this->creator));
			}
			$this->_out('/CreationDate '.$this->_textstring('D:'.date('YmdHis')));
		}
		
		/**
		 * _putcatalog
		 * @access private
		 */
		function _putcatalog() {
			$this->_out('/Type /Catalog');
			$this->_out('/Pages 1 0 R');
			if($this->ZoomMode=='fullpage') {
				$this->_out('/OpenAction [3 0 R /Fit]');
			}
			elseif($this->ZoomMode=='fullwidth') {
				$this->_out('/OpenAction [3 0 R /FitH null]');
			}
			elseif($this->ZoomMode=='real') {
				$this->_out('/OpenAction [3 0 R /XYZ null null 1]');
			}
			elseif(!is_string($this->ZoomMode)) {
				$this->_out('/OpenAction [3 0 R /XYZ null null '.($this->ZoomMode/100).']');
			}
			if($this->LayoutMode=='single') {
				$this->_out('/PageLayout /SinglePage');
			}
			elseif($this->LayoutMode=='continuous') {
				$this->_out('/PageLayout /OneColumn');
			}
			elseif($this->LayoutMode=='two') {
				$this->_out('/PageLayout /TwoColumnLeft');
			}
		}
		
		/**
		 * _puttrailer
		 * @access private
		 */
		function _puttrailer() {
			$this->_out('/Size '.($this->n+1));
			$this->_out('/Root '.$this->n.' 0 R');
			$this->_out('/Info '.($this->n-1).' 0 R');
		}
		
		/**
		 * _enddoc
		 * @access private
		 */
		function _enddoc() {
			$this->_putpages();
			$this->_putresources();
			//Info
			$this->_newobj();
			$this->_out('<<');
			$this->_putinfo();
			$this->_out('>>');
			$this->_out('endobj');
			//Catalog
			$this->_newobj();
			$this->_out('<<');
			$this->_putcatalog();
			$this->_out('>>');
			$this->_out('endobj');
			//Cross-ref
			$o=strlen($this->buffer);
			$this->_out('xref');
			$this->_out('0 '.($this->n+1));
			$this->_out('0000000000 65535 f ');
			for($i=1;$i<=$this->n;$i++) {
				$this->_out(sprintf('%010d 00000 n ',$this->offsets[$i]));
			}
			//Trailer
			$this->_out('trailer');
			$this->_out('<<');
			$this->_puttrailer();
			$this->_out('>>');
			$this->_out('startxref');
			$this->_out($o);
			$this->_out('%%EOF');
			$this->state=3;
		}
		
		/**
		 * _beginpage
		 * @access private
		 */
		function _beginpage($orientation) {
			$this->page++;
			$this->pages[$this->page]='';
			$this->state=2;
			$this->x=$this->lMargin;
			$this->y=$this->tMargin;
			$this->FontFamily='';
			//Page orientation
			if(!$orientation) {
				$orientation=$this->DefOrientation;
			}
			else {
				$orientation=strtoupper($orientation{0});
				if($orientation!=$this->DefOrientation) {
					$this->OrientationChanges[$this->page]=true;
				}
			}
			if($orientation!=$this->CurOrientation) {
				//Change orientation
				if($orientation=='P') {
					$this->wPt=$this->fwPt;
					$this->hPt=$this->fhPt;
					$this->w=$this->fw;
					$this->h=$this->fh;
				}
				else {
					$this->wPt=$this->fhPt;
					$this->hPt=$this->fwPt;
					$this->w=$this->fh;
					$this->h=$this->fw;
				}
				$this->PageBreakTrigger=$this->h-$this->bMargin;
				$this->CurOrientation=$orientation;
			}
		}
		
		/**
		 * End of page contents
		 * @access private
		 */
		function _endpage() {
			$this->state=1;
		}
		
		/**
		 * Begin a new object
		 * @access private
		 */
		function _newobj() {
			$this->n++;
			$this->offsets[$this->n]=strlen($this->buffer);
			$this->_out($this->n.' 0 obj');
		}
		
		/**
		 * Underline text
		 * @access private
		 */
		function _dounderline($x,$y,$txt) {
			$up=$this->CurrentFont['up'];
			$ut=$this->CurrentFont['ut'];
			$w=$this->GetStringWidth($txt)+$this->ws*substr_count($txt,' ');
			return sprintf('%.2f %.2f %.2f %.2f re f',$x*$this->k,($this->h-($y-$up/1000*$this->FontSize))*$this->k,$w*$this->k,-$ut/1000*$this->FontSizePt);
		}
		
		/**
		 * Extract info from a JPEG file
		 * @access private
		 */
		function _parsejpg($file) {
			$a=GetImageSize($file);
			if(!$a) {
				$this->Error('Missing or incorrect image file: '.$file);
			}
			if($a[2]!=2) {
				$this->Error('Not a JPEG file: '.$file);
			}
			if(!isset($a['channels']) or $a['channels']==3) {
				$colspace='DeviceRGB';
			}
			elseif($a['channels']==4) {
				$colspace='DeviceCMYK';
			}
			else {
				$colspace='DeviceGray';
			}
			$bpc=isset($a['bits']) ? $a['bits'] : 8;
			//Read whole file
			$f=fopen($file,'rb');
			$data='';
			while(!feof($f)) {
				$data.=fread($f,4096);
			}
			fclose($f);
			return array('w'=>$a[0],'h'=>$a[1],'cs'=>$colspace,'bpc'=>$bpc,'f'=>'DCTDecode','data'=>$data);
		}
		
		/**
		 * Extract info from a PNG file
		 * @access private
		 */
		function _parsepng($file) {
			$f=fopen($file,'rb');
			if(!$f) {
				$this->Error('Can\'t open image file: '.$file);
			}
			//Check signature
			if(fread($f,8)!=chr(137).'PNG'.chr(13).chr(10).chr(26).chr(10)) {
				$this->Error('Not a PNG file: '.$file);
			}
			//Read header chunk
			fread($f,4);
			if(fread($f,4)!='IHDR') {
				$this->Error('Incorrect PNG file: '.$file);
			}
			$w=$this->_freadint($f);
			$h=$this->_freadint($f);
			$bpc=ord(fread($f,1));
			if($bpc>8) {
				$this->Error('16-bit depth not supported: '.$file);
			}
			$ct=ord(fread($f,1));
			if($ct==0) {
				$colspace='DeviceGray';
			}
			elseif($ct==2) {
				$colspace='DeviceRGB';
			}
			elseif($ct==3) {
				$colspace='Indexed';
			}
			else {
				$this->Error('Alpha channel not supported: '.$file);
			}
			if(ord(fread($f,1))!=0) {
				$this->Error('Unknown compression method: '.$file);
			}
			if(ord(fread($f,1))!=0) {
				$this->Error('Unknown filter method: '.$file);
			}
			if(ord(fread($f,1))!=0) {
				$this->Error('Interlacing not supported: '.$file);
			}
			fread($f,4);
			$parms='/DecodeParms <</Predictor 15 /Colors '.($ct==2 ? 3 : 1).' /BitsPerComponent '.$bpc.' /Columns '.$w.'>>';
			//Scan chunks looking for palette, transparency and image data
			$pal='';
			$trns='';
			$data='';
			do {
				$n=$this->_freadint($f);
				$type=fread($f,4);
				if($type=='PLTE') {
					//Read palette
					$pal=fread($f,$n);
					fread($f,4);
				}
				elseif($type=='tRNS') {
					//Read transparency info
					$t=fread($f,$n);
					if($ct==0) {
						$trns=array(ord(substr($t,1,1)));
					}
					elseif($ct==2) {
						$trns=array(ord(substr($t,1,1)),ord(substr($t,3,1)),ord(substr($t,5,1)));
					}
					else {
						$pos=strpos($t,chr(0));
						if(is_int($pos)) {
							$trns=array($pos);
						}
					}
					fread($f,4);
				}
				elseif($type=='IDAT') {
					//Read image data block
					$data.=fread($f,$n);
					fread($f,4);
				}
				elseif($type=='IEND') {
					break;
				}
				else {
					fread($f,$n+4);
				}
			}
			while($n);
			if($colspace=='Indexed' and empty($pal)) {
				$this->Error('Missing palette in '.$file);
			}
			fclose($f);
			return array('w'=>$w, 'h'=>$h, 'cs'=>$colspace, 'bpc'=>$bpc, 'f'=>'FlateDecode', 'parms'=>$parms, 'pal'=>$pal, 'trns'=>$trns, 'data'=>$data);
		}
		
		/**
		 * Read a 4-byte integer from file
		 * @access private
		 */
		function _freadint($f) {
			$i=ord(fread($f,1))<<24;
			$i+=ord(fread($f,1))<<16;
			$i+=ord(fread($f,1))<<8;
			$i+=ord(fread($f,1));
			return $i;
		}
		
		/**
		 * Format a text string
		 * @access private
		 */
		function _textstring($s) {
			return '('.$this->_escape($s).')';
		}
		
		/**
		 * Add \ before \, ( and )
		 * @access private
		 */
		function _escape($s) {
			return str_replace(')','\\)',str_replace('(','\\(',str_replace('\\','\\\\',$s)));
		}
		
		/**
		 *
		 * @access private
		 */
		function _putstream($s) {
			$this->_out('stream');
			$this->_out($s);
			$this->_out('endstream');
		}
		
		/**
		 * Add a line to the document
		 * @access private
		 */
		function _out($s) {
			if($this->state==2) {
				$this->pages[$this->page].=$s."\n";
			}
			else {
				$this->buffer.=$s."\n";
			}
		}
	} // END OF CLASS
	
	//Handle special IE contype request
	if(isset($HTTP_SERVER_VARS['HTTP_USER_AGENT']) and $HTTP_SERVER_VARS['HTTP_USER_AGENT']=='contype') {
		Header('Content-Type: application/pdf');
		exit;
	}
}
//============================================================+
// END OF FILE
//============================================================+
?>
